unit FTPCacheManager;
{
    UNIT FTPCacheManager
    Version number 1.11(early beta)

This unit contains the FTP cache manager TFTPCache class.
All methods are described in the interface part.

Notes:
    * TFTPCache must re-instanted on every control connection.
    * 3rd of August: version 1.01: Added cmMessages constant.
    * 3rd of August: version 1.02: Conversions to PChar for DeleteFile()
      [API has been changed - ver.1.0 was designed on 3.1API], added
      CacheFileName variable.
    * 3rd of August: version 1.03: IsValidInventory() corrected. Delphi3
      doesn't support write directly to string[index] positions by BlockRead()
      or Move(). (original string substituted by a PChar-like char-array).
    * 3rd of August: version 1.04: Bug fixed in IsValidInventory() - it didn't
      close the inventory file and caused ugly sharing problems.
    * 6th of August: version 1.1: Added username support. Method headers
      changed: RegisterEntry(), LocalFromCache();
    * 6th of August: version 1.11: Kewl bug fixed in CacheEntryRecods;
      hey, noone told me that 32-bit Delphi string are just pointers :))))
      Yeah, I'm just an old Pascal motorcycler :)

Created by Pter Karsai, 11th of July, 3rd, 6th of August '99
}

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, ExtCtrls;

{ cache manager reply constants }
const cmNO_ERROR  = 0;  { operation successfully finished }
      cmIO_ERROR  = 1;  { disk I/O error occured }
      cmNOT_FOUND = 2;  { cache entry not found }
      cmEXPIRED   = 3;  { cache entry expired }
      cmINVALID   = 4;  { invalid file structure or data }
      cmNO_CACHE  = 5;  { cache directory not found }

      cmMessages : array[0..5] of PChar = (
        'No error.', 'Disk I/O error.', 'Cache entry not found.',
        'Cache entry expired.', 'Invalid cache structure or data.',
        'Cache directory not found.');

      CIName : string   = 'cache.inv';  { cache inventory file name }
      CIHeader: string  = 'p$Cache|mssuxx|v0.1$';
      { cache inventory file header }


{ cache inventory file record structure }
type CacheEntryRecord = record
     cerHostName : string[128];  { host name this record assigned to }
     cerUserName : string[32];   { login name this directory assigned to }
     cerDirName  : array[0..1023] of char; { yep... }
                   { directory part this record assigned to }
     cerDate     : TDateTime;    { creation date and time (only date used) }
     cerFileName : string[8];    { filename in system cache directory }
end;

TFTPCache = class(TObject)
private
{ cache data }
   CacheDir      : string;  { system cache directory }
   RemoteHost    : string;  { host this class assigned to }
   ExpDays       : word;    { number of days must spend to expiration }
   CacheFileName : array[1..2048] of char; { full cache file name }
   LastFileSize  : longint; { last file size checked by FileExists() }
   LastItemPos   : longint; { last item's position in CI }

{ ----------------------------------------------------------------------------}
{ private methods }
{ ----------------------------------------------------------------------------}
   function FileExists(FileName: string): boolean;
{ Returns TRUE if given file exists in cache directory, else returns FALSE }

   function IsValidInventory: boolean;
{ Returns TRUE if cache inventory file is valid (header is right), else FALSE }

{ ----------------------------------------------------------------------------}

public
{ ----------------------------------------------------------------------------}
{ cache service methods }
{ ----------------------------------------------------------------------------}
   function RegisterEntry(UserName, DirName: string): string;
{ Function: Make a new entry in the cache inventory, return with the registered
  (unique) filename which will generated by this method. If returning string
  is empty, it means there was an error attempting to register. }

   function LoadFromCache(UserName, CurDir: string; var EntryName: string):byte;
{ Function: Load file name from cache, if CurDir is in the cache. Loaded
  file will saved to EntryName. Returning value can be cmNO_ERROR if cache
  entry found; cmNOT_FOUND if no entry found for the given directory;
  cmEXPIRED if cache data expired. In the last case, EntryName will contain
  the cache entry file name, so cmEXPIRED can be ignored. cmINVALID on ret.,
  if cache inventory file doesn't exist or corrupted. }

   function RefreshLastItem: byte;
{ Function: Refresh the last item's date found by LoadFromCache(). If item
  couldn't found by LoadFromCache(), RefreshLastItem() will do nothing. }

   function GarbageCollector: byte;
{ Function: Runs a garbage collector routine - delete all expired entries.
  Returning value can be cmINVALID, cmNO_CACHE, cmIO_ERROR, cmNO_ERROR }

   function EmptyCache: byte;
{ Function: Delete all cache entries. Returning value as cmX* constants. }

{ ----------------------------------------------------------------------------}
{ constructor and destructor }
{ ----------------------------------------------------------------------------}
   constructor Create(PCacheDir, PRemoteHost: string; PExpDays: word);
{ Parameter PRemoteHost define remote host class entries assigned to.
  PExpDays is the number of days must have spend to expiration.
  Parameter PCacheDir define system cache directory. }

   destructor Destroy; override;
   procedure Free;
end;

implementation
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ private methods -------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
function TFTPCache.FileExists(FileName: string): boolean;
var tempF: file;  { temporary file }
begin
{ assign and try to open }
     Result:= true;
     AssignFile(tempF, FileName);
     try
        Reset(tempF, 1);
     except
        on EInOutError do Result:= false;
     end;

{ close file if exists }
     if Result then
     begin
        LastFileSize:= FileSize(tempF);
        CloseFile(tempF)
     end
     else
        LastFileSize:= 0;
end;

{------------------------------------------------------------------------------}

function TFTPCache.IsValidInventory: boolean;
var tempF   : file;    { temporary file }
    noRead  : integer; { number of readed bytes }
    tempBuf : array[0..64] of char;  { Delphi3's BlockRead() can't write to
                                        string buffer - I dunno why. }
begin
{ check exist of cache inventory }
     Result:= false;
     if FileExists(CacheDir + CIName) then begin
{ assign and try to read header }
        AssignFile(tempF, CacheFileName);
        reset(tempF, 1);
        blockread(tempF, tempBuf, Length(CIHeader), noRead);
{ check validation }
        tempBuf[noRead]:= #0;  { terminate string as PChar }
        if CIHeader = StrPas(@tempBuf) then Result:= true;
     end;
{ close temporary file }
     CloseFile(tempF);
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ cache service methods -------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
function TFTPCache.RegisterEntry(UserName, DirName: string): string;
var cacheFile : file;              { cache inventory file }
    cEntry    : CacheEntryRecord;  { cache entry }
    x         : byte;              { cycle variable }
    trashInt  : integer;           { trash integer for BlockWrite() }
begin
{ initialize random-generator }
     Randomize;
{ repeat while generated name already exists }
     repeat
        SetLength(Result, 8);
        for x:= 1 to 8 do
            Result[x]:= char(ord('A') + random(ord('Z') - ord('A') + 1));
     until not FileExists(Result);

{ if we've got unique filename, register it }
     if IsValidInventory then begin
     { fill record data }
        fillchar(cEntry, sizeof(cEntry), #0); { blow out record }
        cEntry.cerHostName := RemoteHost;     { remote host by constructor }
        cEntry.cerUserName := UserName;       { login name }
        StrCopy(cEntry.cerDirName, PChar(DirName));        { directory name from parameter }
        cEntry.cerDate     := Date;           { current date }
        cEntry.cerFileName := Result;         { generated file name }
     { prepare append }
        AssignFile(cacheFile, CacheDir + CIName);
        try
           reset(cacheFile, 1);  { open file }
           seek(cacheFile, filesize(cacheFile));  { seek to EOF }
           blockwrite(cacheFile, cEntry, sizeof(cEntry), trashInt); { append }
        except
           on EInOutError do Result:= ''; { couldn't register }
        end;
        if Result <> '' then CloseFile(cacheFile); { yep }
     end
     else
        Result:= '';  { if inventory file isn't valid }
end;

{------------------------------------------------------------------------------}

function TFTPCache.LoadFromCache(UserName, CurDir: string;
                                 var EntryName: string): byte;
var cacheFile: file;              { cache inventory file }
    cEntry   : CacheEntryRecord;  { cache entry }
    noRead   : integer;           { number of bytes read }
    entryGot : boolean;           { TRUE if corresponding entry found }
begin
{ we suppose, inventory file is corrupted and we didn't find an item }
     Result:= cmINVALID;
     LastItemPos:= 0;
{ if inventory exists and valid... do the work }
     if not IsValidInventory then exit;

{ open cache inventory }
     AssignFile(cacheFile, CacheDir + CIName);
     reset(cacheFile, 1);
     seek(cacheFile, Length(CIHeader));  { skip header }
     entryGot:= false;
{ read all entries }
     while not eof(cacheFile) and not entryGot do begin
           blockread(cacheFile, cEntry, sizeof(cEntry), noRead);
    { look - if no exact record size, it's a shit, corrupt file }
           if noRead < sizeof(cEntry) then begin
              CloseFile(cacheFile); exit end;
    { check entry's correspondence }
           if (cEntry.cerDirName = CurDir) and (cEntry.cerHostName = remoteHost)
              and (cEntry.cerUserName = UserName) and FileExists(CacheDir +
              cEntry.cerFileName) and (LastFileSize > 0) then
           begin
           { we found it, is it expired? }
              if cEntry.cerDate < Date - ExpDays then
                 Result:= cmEXPIRED else Result:= cmNO_ERROR;
              EntryName:= cEntry.cerFileName; { copy to EntryName }
              entryGot:= true;  { exit from cycle }
              LastItemPos:= FilePos(cacheFile) - sizeof(cEntry);
           end;
     end;
{ if we didn't found such entry... }
     if eof(cacheFile) and not entryGot then Result:= cmNOT_FOUND;
{ but summa summarum: we gotta close cachefile }
     CloseFile(cacheFile);
end;

{------------------------------------------------------------------------------}

function TFTPCache.RefreshLastItem: byte;
var cacheFile: file;              { cache inventory file }
    cEntry   : CacheEntryRecord;  { cache entry }
    noRead   : integer;           { number of bytes read }
begin
{ we suppose, everything's allright }
     Result:= cmNO_ERROR;

{ if inventory doesn't exist or invalid..}
     if not IsValidInventory then
        Result:= cmINVALID;
{ if there's no last item found }
     if LastItemPos = 0 then
        Result:= cmNOT_FOUND;

{ open cache inventory }
     AssignFile(cacheFile, CacheDir + CIName);
     try
        reset(cacheFile, 1);
        seek(cacheFile, LastItemPos);  { goto marker }
     except
        on EInOutError do  { no go to marker }
        begin
            CloseFile(cacheFile);
            Result:= cmIO_ERROR   { we have an error! }
        end;
     end;
{ read entry if we didn't detect error }
     if Result = cmNO_ERROR then
     begin
     { try to read }
         try
           blockread(cacheFile, cEntry, sizeof(cEntry), noRead);
         except
           on EInOutError do  { no go to marker }
           begin
               CloseFile(cacheFile);
               Result:= cmIO_ERROR;   { we have an error! }
               exit
           end;
         end;

     { look - if no exact record size, it's a shit, corrupt file }
        if noRead < sizeof(cEntry) then
        begin
            CloseFile(cacheFile);
            Result:= cmINVALID;
            exit
        end;

     { set date to the present }
        cEntry.cerDate:= Date;

     { try to write }
         try
           blockwrite(cacheFile, cEntry, sizeof(cEntry), noRead);
         except
           on EInOutError do  { no go to marker }
           begin
               CloseFile(cacheFile);
               Result:= cmIO_ERROR;   { we have an error! }
               exit
           end;
         end;
     { if everything's allright... }
         if Result = cmNO_ERROR then
            CloseFile(cacheFile);
     end;
end;

{------------------------------------------------------------------------------}

function TFTPCache.GarbageCollector: byte;
var cacheFile: file;              { cache inventory file }
    tempFile : file;              { temporary store }
    cEntry   : CacheEntryRecord;  { cache entry }
    noRead   : integer;           { number of bytes read }
    cDate    : TDateTime;
begin
     cDate:= Date;
{ does the cache inventory exits? }
     if not FileExists(CacheDir + CIName) then
     begin
          Result:= cmNO_CACHE;
          exit;
     end;

{ we suppose, inventory file is corrupted }
     Result:= cmINVALID;
{ if inventory exists and valid... do the work }
     if not IsValidInventory then exit;

{ open cache inventory }
     AssignFile(cacheFile, CacheDir + CIName);
     reset(cacheFile, 1);
     seek(cacheFile, Length(CIHeader));  { skip header }

{ create temporary file }
     Result:= cmIO_ERROR;
     AssignFile(tempFile, CacheDir + '$tempms.sux');
     try
        rewrite(tempFile, 1); { create and write header }
        blockwrite(tempFile, CIHeader[1], Length(CIHeader), noRead);
     except
        on EInOutError do begin  { close all files and exit }
           CloseFile(cacheFile); CloseFile(tempFile); exit;
        end;
     end;

{ read all entries }
     while not eof(cacheFile) do begin
           blockread(cacheFile, cEntry, sizeof(cEntry), noRead);
    { look - if no exact record size, it's a shit, corrupt file }
           if noRead < sizeof(cEntry) then begin
              CloseFile(cacheFile); CloseFile(tempFile); exit end;
    { is it expired? not? fine! save it to tempFile  }
           if double(cEntry.cerDate) >= (double(cDate) - ExpDays) then
           { save }
              try
                 blockwrite(tempFile, cEntry, sizeof(cEntry), noRead);
              except
                 on EInOutError do begin  { close all files and exit }
                    CloseFile(cacheFile); CloseFile(tempFile); exit; end;
              end;
      end;

{ close all files }
     CloseFile(cacheFile); CloseFile(tempFile);

{ delete cache inventory file and rename tempFile to CIName }
     if not DeleteFile(@CacheFileName) or
        not RenameFile(CacheDir + '$tempms.sux', CacheDir + CIName) then
        exit;

{ if we reached this point, we're happy. operation was successful }
     Result:= cmNO_ERROR;
end;

{------------------------------------------------------------------------------}

function TFTPCache.EmptyCache: byte;
var cacheFile: file;              { cache inventory file }
    cEntry   : CacheEntryRecord;  { cache entry }
    noRead   : integer;           { number of bytes read }
    tempCE   : array[1..2048] of char;  { temporary PChar for DeleteFile() }
begin
{ if cache inventory file doesn't exist, simply exit }
     if not FileExists(CacheDir + CIName) then
     begin
     { if not exists, just create }
          try
             AssignFile(cacheFile, CacheDir + CIName);
             rewrite(cacheFile, 1);  { create }
             blockwrite(cacheFile, CIHeader[1], Length(CIHeader), noRead);
          finally
             CloseFile(cacheFile);
          end;
          Result:= cmNO_ERROR;
          exit;
     end;

{ if exists, scan for available cache items }
     Result:= cmINVALID;  { we suppose, file is has bad structure }
     if IsValidInventory then begin
     { skip header }
        AssignFile(cacheFile, CacheDir + CIName);
        reset(cacheFile, 1);
        seek(cacheFile, Length(CIHeader));
     { read entries }
        while not eof(cacheFile) do begin
              blockread(cacheFile, cEntry, sizeof(cEntry), noRead);
        { look - if no exact record size, it's a shit }
              if noRead < sizeof(cEntry) then begin
                 CloseFile(cacheFile); exit end;
        { we don't care about errors - c'est la vie ;) no exceptions by DF }
              StrPCopy(@tempCE, CacheDir + cEntry.cerFileName);
              DeleteFile(@tempCE)
        end;

     { re-create cache inventory file }
          try
             AssignFile(cacheFile, CacheDir + CIName);
             rewrite(cacheFile, 1);  { create }
             blockwrite(cacheFile, CIHeader[1], Length(CIHeader), noRead);
          finally
             CloseFile(cacheFile);
          end;

     { if we reach this point, everything must allright }
        Result:= cmNO_ERROR;
     end
end;
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ constructor/destructor methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
constructor TFTPCache.Create(PCacheDir, PRemoteHost: string; PExpDays: word);
begin
     inherited Create;
{ save parameters }
     RemoteHost  := PRemoteHost;
     ExpDays     := PExpDays;
     CacheDir    := PCacheDir;
     LastItemPos := 0;  { we didn't found item at first }
     StrPCopy(@CacheFileName, CacheDir + CIName);

{ if cache inventory file doesn't exists, create it}
     if not FileExists(CacheDir + CIName) then EmptyCache;
end;

{------------------------------------------------------------------------------}

destructor TFTPCache.Destroy;
begin
     inherited Destroy;
end;

{------------------------------------------------------------------------------}

procedure TFTPCache.Free;
begin
     if Self <> nil then Destroy;
end;

end.
